# Parametryczne filtry półkowe i szczytowe
# wg Zolzer: DAXF: Digital Audio Effects, 2nd edition, p. 64 & 66

from math import tan, pi, sqrt


s2 = sqrt(2)


def low_shelf(fc, gain, db=False, fs=48000):
    v = get_gain(gain, db)
    if v >= 1:
        return low_boost(fc, v, fs)
    else:
        return low_cut(fc, v, fs)


def high_shelf(fc, gain, db=False, fs=48000):
    v = get_gain(gain, db)
    if v >= 1:
        return high_boost(fc, v, fs)
    else:
        return high_cut(fc, v, fs)


def peak(fc, gain, q, db=False, fs=48000):
    v = get_gain(gain, db)
    if v >= 1:
        return peak_boost(fc, v, q, fs)
    else:
        return peak_cut(fc, v, q, fs)


def get_gain(v, db):
    if db is True or db in ('db', 'dB', 1):
        return 10**(v / 20)
    else:
        return v


def low_boost(fc, v, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    den = 1 + s2 * k + k2
    b0 = 1 + s2 * sqrt(v) * k + v * k2
    b1 = 2 * (v * k2 - 1)
    b2 = 1 - s2 * sqrt(v) * k + v * k2
    a1 = 2 * (k2 - 1)
    a2 = 1 - s2 * k + k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


def low_cut(fc, v, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    den = v + s2 * sqrt(v) * k + k2
    b0 = v * (1 + s2 * k + k2)
    b1 = 2 * v * (k2 - 1)
    b2 = v * (1 - s2 * k + k2)
    a1 = 2 * (k2 - v)
    a2 = v - s2 * sqrt(v) * k + k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


def high_boost(fc, v, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    den = 1 + s2 * k + k2
    b0 = v + s2 * sqrt(v) * k + k2
    b1 = 2 * (k2 - v)
    b2 = v - s2 * sqrt(v) * k + k2
    a1 = 2 * (k2 - 1)
    a2 = 1 - s2 * k + k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


def high_cut(fc, v, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    den = 1 + s2 * sqrt(v) * k + v * k2
    b0 = v * (1 + s2 * k + k2)
    b1 = 2 * v * (k2 - 1)
    b2 = v * (1 - s2 * k + k2)
    a1 = 2 * (v * k2 - 1)
    a2 = 1 - s2 * sqrt(v) * k + v * k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


def peak_boost(fc, v, q, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    qr = 1 / q
    den = 1 + qr * k + k2
    b0 = 1 + v * qr * k + k2
    b1 = 2 * (k2 - 1)
    b2 = 1 - v * qr * k + k2
    a1 = 2 * (k2 - 1)
    a2 = 1 - qr * k + k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


def peak_cut(fc, v, q, fs=48000):
    k = tan(pi * fc / fs)
    k2 = k**2
    qr = 1 / q
    vr = 1 / v
    den = 1 + vr * qr * k + k2
    b0 = 1 + qr * k + k2
    b1 = 2 * (k2 - 1)
    b2 = 1 - qr * k + k2
    a1 = 2 * (k2 - 1)
    a2 = 1 - vr * qr * k + k2
    b = [b0 / den, b1 / den, b2 / den]
    a = [1, a1 / den, a2 / den]
    return b, a


if __name__ == "__main__":
    from matplotlib import pyplot as plt
    import scipy.signal as sig
    import numpy as np

    # filtry półkowe

    db_gain = (12, 8, 4, -4, -8, -12)
    sh_low = [low_shelf(100, v, db=True) for v in db_gain]
    sh_high = [high_shelf(3000, v, db=True) for v in db_gain]

    for filter in (sh_low, sh_high):
        for i, (b, a) in enumerate(filter):
            f, hf = sig.freqz(b, a, worN=2048, fs=48000)
            filter[i] = (f, 20 * np.log10(np.abs(hf)))

    fig1, ax1 = plt.subplots(figsize=(8, 4), tight_layout=True)
    for filter in (sh_low, sh_high):
        for (f, hf) in filter:
            ax1.semilogx(f, hf)
    ax1.grid()
    ax1.set_xlabel('Częstotliwość [Hz]')
    ax1.set_ylabel('Wzmocnienie [dB]')
    ax1.set_title('Charakterystyki filtrów półkowych (zmienne wzmocnienie/tłumienie)')

    pk = [peak(500, v, 1.1, db=True) for v in db_gain]

    fig2, ax2 = plt.subplots(figsize=(8, 4), tight_layout=True)
    for (b, a) in pk:
        f, hf = sig.freqz(b, a, worN=2048, fs=48000)
        ax2.semilogx(f, 20 * np.log10(np.abs(hf)))
    ax2.grid()
    ax2.set_xlabel('Częstotliwość [Hz]')
    ax2.set_ylabel('Wzmocnienie [dB]')
    ax2.set_title('Charakterystyki filtrów szczytowych (zmienne wzmocnienie, stała dobroć)')

    # filtry szczytowe

    q = (0.5, 0.8, 1.1, 1.4, 1.7)
    pk2 = [peak(500, v, qq, db=True) for v in (12, -12) for qq in q]

    fig3, ax3 = plt.subplots(figsize=(8, 4), tight_layout=True)
    for (b, a) in pk2:
        f, hf = sig.freqz(b, a, worN=2048, fs=48000)
        ax3.semilogx(f, 20 * np.log10(np.abs(hf)))
    ax3.grid()
    ax3.set_xlabel('Częstotliwość [Hz]')
    ax3.set_ylabel('Wzmocnienie [dB]')
    ax3.set_title('Charakterystyki filtrów szczytowych (stałe wzmocnienie, zmienna dobroć)')

    # bank filtrów oktawowych

    bands = [1000 * 2**i for i in range(-5, 5)]
    q = sqrt(2) / (sqrt(2) - 1)
    octave = [peak(f, v, q, db=True) for v in (12, -12) for f in bands]

    fig4, ax4 = plt.subplots(figsize=(8, 4), tight_layout=True)
    for (b, a) in octave:
        f, hf = sig.freqz(b, a, worN=4800, fs=48000)
        ax4.semilogx(f, 20 * np.log10(np.abs(hf)))
    ax4.set_xlim(20, 24000)
    ax4.grid()
    ax4.set_xlabel('Częstotliwość [Hz]')
    ax4.set_ylabel('Wzmocnienie [dB]')
    ax4.set_title('Charakterystyki filtrów oktawowych')

    plt.show()
