"""Filtr różniczkujący"""

import numpy as np
import scipy.signal as sig
from matplotlib import pyplot as plt

fs = 48000

# idealny filtr Hilberta: H(z) = -j (dodatnie f), j (ujemne f), 0 (w 0)
# antysymetryczna i nieparzysta odp. impulsowa

N = 101
eps = 1e-5
h = -sig.firwin2(N, [0, eps, 1 - eps, 1], [0, 1, 1, 0],
        window='hamming', antisymmetric=True)

w, hf = sig.freqz(h, fs=fs, whole=True)
w2 = w - fs / 2
ideal = np.ones_like(w)

fig, ax = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
ax[0].plot(w2, np.abs(ideal), ls='--', c='#a0a0a0', label='idealna')
ax[0].plot(w2, np.fft.fftshift(np.abs(hf)), label='zaprojektowana')
ax[0].set_ylabel('Wzmocnienie (lin)')
ax[0].grid()
ax[0].legend()
ax[0].set_title(f'Charakterystyka amplitudowa filtru Hilberta N={N}')

# phase corrected for the shift
shift = (N - 1) // 2
norm = np.exp(2j * np.pi * w / fs)**(-shift)

ax[1].plot(w2, np.degrees(np.fft.fftshift(np.angle(hf / norm))))
ax[1].set_xlabel('Częstotliwość [Hz]')
ax[1].set_ylabel('Faza [°]')
ax[1].set_ylim(-120, 120)
ax[1].grid()
ax[1].set_title(f'Charakterystyka fazowa filtru Hilberta N={N}')

plt.figure(figsize=(8, 4))
plt.stem(h, markerfmt='.', use_line_collection=True)
plt.xlabel('Nr próbki')
plt.ylabel('Wartość')
plt.title(f'Odpowiedź impulsowa filtru Hilberta N={N}')


# sygnał - fala piłokształtna
x = sig.sawtooth(2 * np.pi * np.arange(2100) * 100 / fs)
# x = np.sin(2 * np.pi * np.arange(4000) * 1000 / fs)
# x -= np.mean(x)
# sygnał po transformacji Hilberta (obcięcie stanu nieustalonego)
a_im = sig.lfilter(h, 1, x)[shift:]
# sygnał analityczny
a = x[:2048] + 1j * a_im[:2048]
# a = sig.hilbert(x[:2048])
w_a = np.fft.fft(a * np.hamming(2048))
w_x = np.fft.fft(x[:2048] * np.hamming(2048))

plt.figure(figsize=(8, 4))
fr = np.fft.fftshift(np.fft.fftfreq(2048, 1 / fs))
plt.plot(fr, 20 * np.log10(np.abs(np.fft.fftshift(w_x)) / 2048),
        c='#c0c0c0', label='Oryginalny')
plt.plot(fr, 20 * np.log10(np.abs(np.fft.fftshift(w_a)) / 2048),
        label='Analityczny')
plt.grid()
plt.legend()
plt.xlabel('Częstotliwość [Hz]')
plt.ylabel('Poziom widma [dB]')
plt.title('Charakterystyka amplitudowa sygnału oryginalnego i analitycznego')

# analiza sygnału mowy
speech = np.load('speech.npy')
speech = speech / 32768.0
speech_im = sig.lfilter(h, 1, speech)[shift:]
speech_an = speech[:-shift] + 1j * speech_im
env = np.abs(speech_an)

smooth = 301
hdp = sig.firwin(smooth, 100, window='hamming', pass_zero='lowpass', fs=16000)
env2 = sig.lfilter(hdp, 1, env)[(smooth - 1) // 2:]
# env2 = np.convolve(env, np.ones(smooth) / smooth)[:len(env)]

fig4, ax4 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
ax4[0].plot(speech, label='sygnał mowy')
ax4[0].plot(env2, label='wygładzona obwiednia')
ax4[1].plot(env, label='moduł sygnału analitycznego')
for a in ax4:
    a.grid()
    a.legend()
    a.set_ylabel('Amplituda')
ax4[-1].set_xlabel('Nr próbki')

plt.show()
