"""Typy charakterystyk filtrów IIR"""

import numpy as np
import scipy.signal as sig
import matplotlib
import matplotlib.pyplot as plt

matplotlib.rcParams['figure.figsize'] = (8, 4)

fs = 48000


def plot_response(w, hf, hfb=None, label=''):
    fig, ax = plt.subplots(tight_layout=True)
    ax.axvline(3000, lw=1, ls=(0, (10, 5)), c='k')
    if hfb is not None:
        ax.plot(w, hfb, c='#c0c0c0', label='Butterworth')
    ax.plot(w, hf, label=label)
    ax.grid()
    ax.legend()
    ax.set_xlim(0, 12000)
    ax.set_ylim(-120, 5)
    ax.set_xlabel('Częstotliwość [Hz]')
    ax.set_ylabel('Wzmocnienie [dB]')
    # ax.set_title('Charakterystyka częstotliwościowa filtru IIR')
    return ax


fc = 3000
fz = 4500
rp = 1
rs = 60
N = 8

# filtr Butterwortha
bb, ab = sig.iirfilter(N, fc, btype='lowpass', ftype='butter', fs=fs)
wb, hb = sig.freqz(bb, ab, 2048, fs=fs)
hbd = 20 * np.log10(np.abs(hb))
plot_response(wb, hbd, None, 'Butterworth')

# filtr Czebyszewa I
bc1, ac1 = sig.iirfilter(N, fc, btype='lowpass', ftype='cheby1', rp=rp, fs=fs)
wc1, hc1 = sig.freqz(bc1, ac1, 2048, fs=fs)
hc1d = 20 * np.log10(np.abs(hc1))
plot_response(wc1, hc1d, hbd, 'Czebyszew I')

# filtr Czebyszewa II
bc2, ac2 = sig.iirfilter(N, fz, btype='lowpass', ftype='cheby2', rs=rs, fs=fs)
wc2, hc2 = sig.freqz(bc2, ac2, 2048, fs=fs)
hc2d = 20 * np.log10(np.abs(hc2))
plot_response(wc2, hc2d, hbd, 'Czebyszew II')

# filtr Eliptyczny
be, ae = sig.iirfilter(N, fc, btype='lowpass', ftype='ellip', rp=rp, rs=rs, fs=fs)
we, he = sig.freqz(be, ae, 2048, fs=fs)
hed = 20 * np.log10(np.abs(he))
plot_response(we, hed, hbd, 'Eliptyczny')

# wszystko razem
ax = plot_response(wb, hbd, None, 'Butterworth')
ax.plot(wc1, hc1d, label='Czebyszew I')
ax.plot(wc2, hc2d, label='Czebyszew II')
ax.plot(we, hed, label='Eliptyczny')
ax.legend()

# filtry eliptyczne różnych rzędów
fig, ax = plt.subplots(tight_layout=True)
ax.axvline(3000, lw=1, ls=(0, (10, 5)), c='k')
for nb in (2, 6, 10, 14):
    b, a = sig.iirfilter(nb, fc, btype='lowpass', ftype='ellip', rp=rp, rs=rs, fs=fs)
    w, h = sig.freqz(b, a, 2048, fs=fs)
    hd = 20 * np.log10(np.abs(h))
    ax.plot(w, hd, label=f'N={nb}')
ax.grid()
ax.legend()
ax.set_xlim(0, 12000)
ax.set_ylim(-120, 5)
ax.set_xlabel('Częstotliwość [Hz]')
ax.set_ylabel('Wzmocnienie [dB]')

# charakterystyka fazowa
fig, ax = plt.subplots(2, 2, tight_layout=True, figsize=(8, 6))
for a in ax.ravel():
    a.axvline(3000, lw=1, ls=(0, (10, 5)), c='k')
    a.grid()
    a.set_xlim(0, 8000)
    a.set_ylim(-180, 180)
ax[0][0].plot(wb, np.degrees(np.angle(hb)))
ax[0][0].set_title('Butterworth')
ax[0][1].plot(wc1, np.degrees(np.angle(hc1)))
ax[0][1].set_title('Czebyszew I')
ax[1][0].plot(wc2, np.degrees(np.angle(hc2)))
ax[1][0].set_title('Czebyszew II')
ax[1][1].plot(we, np.degrees(np.angle(he)))
ax[1][1].set_title('Eliptyczny')
for a in ax[1]:
    a.set_xlabel('Częstotliwość [Hz]')
for a in (ax[0][0], ax[1][0]):
    a.set_ylabel('Faza [°]')

# opóźnienie grupowe
fig, ax = plt.subplots(2, 2, tight_layout=True, figsize=(8, 6))
for a in ax.ravel():
    a.axvline(3000, lw=1, ls=(0, (10, 5)), c='k')
    a.grid()
    a.set_xlim(0, 8000)
ax[0][0].plot(*sig.group_delay((bb, ab), 2048, fs=fs))
ax[0][0].set_title('Butterworth')
ax[0][1].plot(*sig.group_delay((bc1, ac1), 2048, fs=fs))
ax[0][1].set_title('Czebyszew I')
ax[1][0].plot(*sig.group_delay((bc2, ac2), 2048, fs=fs))
ax[1][0].set_title('Czebyszew II')
ax[1][1].plot(*sig.group_delay((be, ae), 2048, fs=fs))
ax[1][1].set_title('Eliptyczny')
for a in ax[1]:
    a.set_xlabel('Częstotliwość [Hz]')
for a in (ax[0][0], ax[1][0]):
    a.set_ylabel('Opóźnienie [próbki]')

plt.show()
