"""Projektowanie filtru FIR różnymi metodami"""

import numpy as np
import scipy.signal as sig
import matplotlib
import matplotlib.pyplot as plt

matplotlib.rcParams['figure.figsize'] = (8, 4)

fs = 48000


def get_signal(n=4096):
    """Sygnał testowy - 5 sinusów i szum."""
    t = np.arange(n) / fs
    fr = np.array([500, 1000, 1500, 2000, 2500]).reshape(-1, 1)
    x = np.sum(np.sin(2 * np.pi * t * fr), axis=0)
    x = x + 0.05 * np.random.randn(len(x))
    x = x / np.max(np.abs(x))
    return x


def plot_signal_spectrum(x, **kwargs):
    """Wykres widma sygnału."""
    fig, ax = plt.subplots(tight_layout=True, **kwargs)
    spectrum = np.fft.rfft(x[:2048] * np.hamming(2048))
    spdb = 20 * np.log10(np.abs(spectrum) / 1024)
    f = np.fft.rfftfreq(2048, 1 / fs)
    ax.plot(f, spdb)
    ax.grid()
    ax.set_ylim(bottom=-80)
    ax.set_xlabel('Częstotliwość [Hz]')
    ax.set_ylabel('Poziom [dB]')
    return ax


def plot_freq_resp(h):
    """Wykres charakterystyki częstotliwościowej filtru."""
    w, hf = sig.freqz(h, worN=2048, fs=fs)
    fig, ax = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
    ax[0].plot(w, np.abs(hf))
    ax[0].set_ylabel('Wzmocnienie (lin)')
    ax[1].plot(w, 20 * np.log10(np.abs(hf)))
    ax[1].set_ylabel('Wzmocnienie [dB]')
    ax[1].set_xlabel('Częstotliwość [Hz]')
    ax[0].grid()
    ax[1].grid()
    return ax


def plot_filter(h, x):
    """Wykres charakterystyki częstotliwościowej filtru i widma po filtracji."""
    w, hf = sig.freqz(h, worN=2048, fs=fs)
    y = sig.lfilter(h, 1, x)
    lag = (len(h) - 1) // 2
    x_sp = np.fft.rfft(x[:2048] * np.hamming(2048))
    x_spdb = 20 * np.log10(np.abs(x_sp) / 1024)
    y_sp = np.fft.rfft(y[lag:lag + 2048] * np.hamming(2048))
    y_spdb = 20 * np.log10(np.abs(y_sp) / 1024)
    f = np.fft.rfftfreq(2048, 1 / fs)
    fig, ax = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
    ax[0].plot(w, 20 * np.log10(np.abs(hf)))
    ax[0].set_ylabel('Wzmocnienie [dB]')
    ax[0].set_title('Charakterystyka częstotliwościowa filtru')
    ax[0].grid()
    # ax[0].set_ylim(bottom=-90)
    ax[1].plot(f, x_spdb, c='#a0a0a0', label='Oryginalny')
    ax[1].plot(f, y_spdb, label='Po filtracji')
    ax[1].set_ylabel('Poziom [dB]')
    ax[1].set_title('Widmo sygnału')
    ax[1].grid()
    ax[1].set_ylim(bottom=-90)
    ax[1].set_xlabel('Częstotliwość [Hz]')
    return ax


x = get_signal()
ax0 = plot_signal_spectrum(x)
ax0.set_xlim(0, 4000)

# Projekt 1
# metoda okienkowania, projekt w dziedzinie czasu
# filtr GP 1250 Hz, p. przejściowe 200 Hz
# obliczona długość: N >= 3,3 * (48000 / 200) = 792
h1 = sig.firwin(801, 1250, window='hamming', pass_zero='highpass', fs=fs)
ax1 = plot_filter(h1, x)
ax1[0].set_ylim(bottom=-80)
ax1[1].set_xlim(0, 4000)


# Projekt 1a - okno Kaisera, min. tłumienie 60 dB
nk, beta = sig.kaiserord(60, 2 * 200 / 48000)
if nk % 2 == 0:
    nk += 1  # dla GP długość musi być nieparzysta
# print(nk, beta)
h1a = sig.firwin(nk, 1250, window=('kaiser', beta), pass_zero='highpass', fs=fs)
ax1a = plot_filter(h1a, x)
ax1a[0].set_ylim(bottom=-100)
ax1a[1].set_xlim(0, 4000)


# Projekt 2 - filtr PP 750-1750 Hz, p. przejściowe 20 Hz
# metoda okienkowania, projekt w dziedzinie częstotliwości
h2 = sig.firwin2(801, (0, 730, 750, 1750, 1770, fs / 2), (0, 0, 1, 1, 0, 0),
                 window='hamming', fs=fs)
ax2 = plot_filter(h2, x)
ax2[0].set_ylim(bottom=-80)
ax2[1].set_xlim(0, 4000)


# Projekt 3 - filtr PZ usuwający środkowy prążek, metoda LS
h3 = sig.firls(801, (0, 1290, 1490, 1510, 1710, fs / 2), (1, 1, 0, 0, 1, 1), fs=fs)
ax3 = plot_filter(h3, x)
ax3[0].set_ylim(bottom=-80)
ax3[1].set_xlim(0, 4000)


# Projekt 4 - filtr PZ usuwający prążki 2 i 4, metoda P-M
h4 = sig.remez(801, (0, 890, 990, 1010, 1110, 1890, 1990, 2010, 2110, fs / 2),
        (1, 0, 1, 0, 1), fs=fs)
# h4 = sig.remez(801, (0, 890, 990, 1010, 1110, 1890, 1990, 2010, 2110, fs / 2),
#         (1, 0.1, 1, 0.1, 1), fs=fs)
ax4 = plot_filter(h4, x)
ax4[0].set_ylim(bottom=-80)
ax4[1].set_xlim(0, 4000)


plt.show()
