"""Projektowanie filtru FIR metodą okienkowania"""

import numpy as np
import scipy.signal as sig
import matplotlib
import matplotlib.pyplot as plt

matplotlib.rcParams['figure.figsize'] = (8, 4)

fs = 48000
N = 101  # długość filtru

# idealna charakterystyka
f = np.fft.rfftfreq(2048, 1 / fs)
plt.figure()
ideal = np.ones_like(f)
ideal[f > 3000] = 0
plt.plot(f, ideal)
plt.xlim(0, 12000)
plt.ylim(-0.1, 1.1)
plt.xlabel('Częstotliwość [Hz]')
plt.ylabel('Wzmocnienie')
plt.title('Idealny filtr DP 3 kHz')

# odpowiedzi impulsowe - długa i przycięta
n_long = 2047
h_long = sig.firwin2(n_long, [0, 3000, 3000, fs / 2], [1, 1, 0, 0], n_long + 1, None, fs=fs)
n_short = 101
h_short = sig.firwin2(n_short, [0, 3000, 3000, fs / 2], [1, 1, 0, 0], n_short + 1, None, fs=fs)
h_short2 = np.zeros_like(h_long)
shift = (n_long - n_short) // 2
h_short2[shift:shift + len(h_short)] = h_short
idx = np.arange(n_long) - (n_long - 1) // 2
win = np.zeros_like(h_long)
win[shift:shift + len(h_short)] = np.max(h_long)

fig2, ax2 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(10, 5))
ax2[0].plot(idx, win, lw=1, c='#c0c0c0')
ax2[0].plot(idx, h_long)
ax2[1].plot(idx, h_short2)
# ax2[0].plot(idx, h_long, '-o', ms=2, lw=1)
# ax2[1].plot(idx, h_short2, '-o', ms=2, lw=1)
ax2[1].set_xlabel('Nr próbki')
ax2[0].set_title('Odpowiedzi impulsowe - pełna i przycięta')

# charakterystyki częstotliwościowe
w, hf = sig.freqz(h_short, worN=2048, fs=48000)
hfdb = 20 * np.log10(np.abs(hf))
fig3, ax3 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
ax3[0].plot(f, ideal, c="#808080")
ax3[0].plot(w, np.abs(hf))
ax3[0].set_ylabel('Wzmocnienie (lin)')
ax3[1].plot(f, 20 * np.log10(np.maximum(ideal, 1e-8)), c="#808080")
ax3[1].plot(w, 20 * np.log10(np.abs(hf)))
ax3[1].set_ylim(bottom=-80)
ax3[1].set_xlim(0, 12000)
ax3[1].set_ylabel('Wzmocnienie [dB]')
ax3[1].set_xlabel('Częstotliwość [Hz]')
ax3[0].set_title('Charakterystyki widmowe filtru')

# nieciągłość odpowiedzi impulsowej
plt.figure(4)
plt.plot(idx, h_short2, '-o', ms=2, lw=1)
plt.xlim(-(n_short // 2) - 15, (n_short // 2) + 15)
plt.xlabel('Nr próbki')
plt.title('Odpowiedź impulsowa po przycięciu')

# okno Blackmana
h_win = sig.firwin2(n_short, [0, 3000, 3000, fs / 2], [1, 1, 0, 0], n_short + 1,
        'blackman', fs=fs)
h_win2 = np.zeros_like(h_long)
h_win2[shift:shift + len(h_win)] = h_win

# porównanie - odpowiedzi impulsowe
fig5, ax5 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
ax5[0].plot(idx, h_short2, '-o', ms=2, lw=1, label='Bez okna')
ax5[1].plot(idx, h_win2, '-o', ms=2, lw=1, label='Okno Blackmana')
ax5[1].set_xlim(-(n_short // 2) - 15, (n_short // 2) + 15)
ax5[0].legend(loc='upper right')
ax5[1].legend(loc='upper right')
ax5[1].set_xlabel('Nr próbki')
ax5[0].set_title('Odpowiedzi impulsowe')

# porównanie - charakterystyki częstotliwościowe
_, hfw = sig.freqz(h_win, worN=2048, fs=48000)
fig6, ax6 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 5))
ax6[0].plot(w, np.abs(hf), c='#808080', label='bez okna')
ax6[0].plot(w, np.abs(hfw), label='okno Blackmana')
ax6[0].set_ylabel('Wzmocnienie (lin)')
ax6[1].plot(w, 20 * np.log10(np.abs(hf)), c='#808080', label='bez okna')
ax6[1].plot(w, 20 * np.log10(np.abs(hfw)), label='okno Blackmana')
ax6[1].set_ylim(bottom=-120)
ax6[1].set_xlim(0, 12000)
for a in ax6:
    a.legend(loc='upper right')
    a.grid()
ax6[1].set_ylabel('Wzmocnienie [dB]')
ax6[1].set_xlabel('Częstotliwość [Hz]')
ax6[0].set_title('Charakterystyki widmowe filtru')

# porównanie - pasmo przepustowe
plt.figure(7)
plt.axvline(3000, lw=1, ls='--', c='k')
plt.plot(w, np.abs(hf), c='#808080', label='bez okna')
plt.plot(w, np.abs(hfw), label='okno Blackmana')
plt.xlim(0, 5000)
plt.ylim(bottom=0.5)
plt.grid()
plt.legend()
plt.ylabel('Wzmocnienie (lin)')
plt.xlabel('Częstotliwość [Hz]')
plt.title('Zniekształcenia w paśmie przepustowym')

# porównanie - pasmo zaporowe
plt.figure(8)
plt.axvline(3000, lw=1, ls='--', c='k')
plt.plot(w, 20 * np.log10(np.abs(hf)), c='#808080', label='bez okna')
plt.plot(w, 20 * np.log10(np.abs(hfw)), label='okno Blackmana')
plt.ylim(-140, 0)
plt.xlim(3000, 12000)
plt.grid()
plt.legend()
plt.ylabel('Wzmocnienie [dB]')
plt.xlabel('Częstotliwość [Hz]')
plt.title('Zniekształcenia w paśmie zaporowym')


# porównanie filtrów o różnej długości
fig9, ax9 = plt.subplots(2, sharex=True, tight_layout=True, figsize=(8, 6))
ax9[0].axvline(3000, lw=1, c='k', ls='--')
ax9[1].axvline(3000, lw=1, c='k', ls='--')
for n in (31, 51, 101, 301, 501):
    h = sig.firwin2(n, [0, 3000, 3000, fs / 2], [1, 1, 0, 0], n + 1,
        'blackman', fs=fs)
    w, hf = sig.freqz(h, worN=2048, fs=48000)
    ax9[0].plot(w, np.abs(hf), label=f'N={n}')
    ax9[1].plot(w, 20 * np.log10(np.abs(hf)), label=f'N={n}')
ax9[1].set_ylim(bottom=-180)
ax9[1].set_xlim(0, 12000)
for a in ax9:
    a.legend(loc='lower left')
    a.grid()
ax9[0].set_ylabel('Wzmocnienie (lin)')
ax9[1].set_ylabel('Wzmocnienie [dB]')
ax9[1].set_xlabel('Częstotliwość [Hz]')
ax9[0].set_title('Charakterystyki widmowe filtru')


plt.show()
